function printErrMsg($msg)
{
    if (![String]::IsNullOrEmpty($msg))
    {
        Write-Host "$msg" -ForegroundColor Red
    }
    else
    {
        $errMsg = "`t0x{0:x} " -f $_.Exception.HResult
        switch ($_.Exception.HResult)
        {
            0x80004001 {$errMsg = $errMsg + "E_NOTIMPL : Not implemented"; break}
            0x80004002 {$errMsg = $errMsg + "E_NOINTERFACE : No such interface supported"; break}
            0x80004003 {$errMsg = $errMsg + "E_POINTER : Pointer that is not valid"; break}
            0x80004004 {$errMsg = $errMsg + "E_ABORT : Operation aborted"; break}
            0x80004005 {$errMsg = $errMsg + "E_FAIL : Unspecified failure"; break}
            0x8000FFFF {$errMsg = $errMsg + "E_UNEXPECTED : Unexpected failure"; break}
            0x80070005 {$errMsg = $errMsg + "E_ACCESSDENIED : General access denied error"; break}
            0x80070006 {$errMsg = $errMsg + "E_HANDLE : Handle that is not valid"; break}
            0x8007000E {$errMsg = $errMsg + "E_OUTOFMEMORY : Failed to allocate necessary memory"; break}
            0x80070057 {$errMsg = $errMsg + "E_INVALIDARG : One or more arguments are not valid"; break}
            Default    {$errMsg = $errMsg + "UNKNOWN ERROR"}
        }
        $errMsg = $errMsg + "`n"
        Write-Host "$errMsg" -ForegroundColor Red
    }
}

try {
    $app = [Runtime.InteropServices.Marshal]::GetActiveObject('eCADSTAR.PCBEditor.Application')
}
catch {
    printErrMsg
}
if ($app -eq $null) {
    $startPCB = Read-Host "No runnning instance of PCB editor found. Press Y to start a new instance of PCB Editor"
    if (($startPCB -eq 'Y') -or ($startPCB -eq 'y')) {
        $app = New-Object -ComObject 'eCADSTAR.PCBEditor.Application'
        $app.visible = $true
    }
}

if ($app -ne $null) {
    $checkCloseDesign = $false

    $designs = $app.Designs
    if ($designs.count -eq 0) {
        $loadDesign = Read-host "No designs open are open in PCD Editor. Press 'O' to open a design"
        if (($loadDesign -eq 'O') -or ($loadDesign -eq 'o')) {
            $designPath = Read-Host "Please enter the full path to the design you want to open"
            try {
                $app.OpenDesign($designPath)
                $checkCloseDesign = $true
            }
            catch {
                printErrMsg
            }
        }
    }
    if ($designs.count -gt 1) {
        $designName = Read-Host "More than one design is open in PCB Editor. Press enter to use the currently active design or enter the name of the design you want to use"
        foreach($design in $designs) {
            if ($design.Name() -eq $designName) {
                $design.active = $true
            }
        }
    }

    try {
        $design = $app.CurrentDesign()
        if ($design -ne $null) {
            Write-Host "Using design $($design.name())"
            Write-Host ""
            $variants = $design.Variants
            if ($variants.count -gt 1) {
                $variantName = Read-Host "The selected design has more than one variant. Press enter to use the currently active variant or enter the name of the variant you want to use"
                foreach($variant in $variants)
                {
                    if ($variant.Name() -eq $variantName) {
                        $variant.active = $true
                    }
                }
                foreach($variant in $variants)
                {
                    if ($variant.active -eq $true) {
                        Write-Host "Using variant $($variant.name())"
                        Write-Host ""
                    }
                }
            }

            $propertyDefinitions = $design.propertyDefinitions
            $outputCompPropDefs = Read-host "Press Y to output a list of component properties"
            if (($outputCompPropDefs -eq 'Y') -or ($outputCompPropDefs -eq 'y')) {
                $compPropertyDefinitions = $propertyDefinitions.component
                write-host "Component property definitions"
                foreach($propertyDefinition in $compPropertyDefinitions) {
                    write-host "Property name: $($propertyDefinition.name), System property: $($propertyDefinition.isSystem), Read only: $($propertyDefinition.isReadOnly)"
                }
                Write-Host ""
            }

            $outputPartPropDefs = Read-host "Press Y to output a list of part properties"
            if (($outputPartPropDefs -eq 'Y') -or ($outputPartPropDefs -eq 'y')) {
                $partPropertyDefinitions = $propertyDefinitions.part
                write-host "Part property definitions"
                foreach($propertyDefinition in $partPropertyDefinitions) {
                    write-host "Property name: $($propertyDefinition.name), System property: $($propertyDefinition.isSystem), Read only: $($propertyDefinition.isReadOnly)"
                }
                Write-Host ""
            }
            
            $components = $design.components
            foreach ( $component in $components) {
                $refDes = $component.ReferenceDesignator

                $description = "Description not found"
                $partName = "Part name not found"
                $part = $component.part
                if ($part -ne $null) {
                    $partName = $part.name()
                    
                    $partProperties = $part.properties
                    $property = $partProperties.find("Description")
                    if ($property -ne $null) {
                        $description = $property.Value()
                    }
                }

                Write-Host "$($refDes), $($partName), $($description)"
            }
            
            if ( $checkCloseDesign) {
                $closeDesign = Read-host "Press Y to close the opened design"
                if (($closeDesign -eq 'Y') -or ($closeDesign -eq 'y')) {
                    $design.close()
                }
            }
        }
    }
    catch {
        printErrMsg
        Write-host "No designs open in PCB editor."
    }

    $closeApp = Read-host "Press Y to close PCB Editor"
    if (($closeApp -eq 'Y') -or ($closeApp -eq 'y')) {
        $app.quit()
    }

}
